<?php
/**
 *
 * @since 1.0.0
 */
use Elementor\Repeater;
use Elementor\Group_Control_Css_Filter;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;

defined( 'ABSPATH' ) || die();

class Rsaddon_Elementor_pro_Loan_Calculator_Widget extends \Elementor\Widget_Base {

	/**
	 * Get widget name.
	 *
	 * Retrieve counter widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'rs-loan-calculator';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve counter widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'RS Loan Calculator', 'rsaddon' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve counter widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'rs-badge';
	}

	public function get_keywords() {
		return [ 'calculator', 'cost', 'measure', 'loan' ];
	}

	/**
	 * Retrieve the list of scripts the counter widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.3.0
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_categories() {
        return [ 'rsaddon_category' ];
    }
	/**
	 * Register services widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */

	// Register Controls
	protected function register_controls() {
		$this->register_loan_calculator_content_controls();
		$this->register_loan_calculator_wrapper_style_controls();
		$this->register_loan_calculator_range_slider_style_controls();
		$this->register_loan_calculator_summery_part_style_controls();
		$this->register_loan_calculator_chart_part_style_controls();
	}

	// Content Dynamic Input Start
	protected function register_loan_calculator_content_controls()
    {
		$this->start_controls_section(
			'section_loan_calculator_content',
			[
				'label' => esc_html__( 'Calculator', 'rsaddon' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);
		// Price based area start
		$this->add_control(
			'currency',
			[
				'label'       => esc_html__( 'Currency Sign', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( '$', 'rsaddon' ),
				'default' => '$',
			]
		);

		// Input Labels
		$this->add_control(
			'value_input_area_title_options',
			[
				'label' => esc_html__( 'Value Input Items Label', 'rsaddon' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		$this->add_control(
			'in_amount_title',
			[
				'label'       => esc_html__( 'Amount Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Amount Title Here..', 'rsaddon' ),
				'default' => 'Amount',
			]
		);
		$this->add_control(
			'in_duration_title',
			[
				'label'       => esc_html__( 'Duration Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Duration Title Here..', 'rsaddon' ),
				'default' => 'Length',
			]
		);
		$this->add_control(
			'in_interest_title',
			[
				'label'       => esc_html__( 'Interest % Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Interest % Title Here..', 'rsaddon' ),
				'default' => 'Interest Rate',
			]
		);

		// Summery Titles
		$this->add_control(
			'summery_area_title_options',
			[
				'label' => esc_html__( 'Summery Area Titles', 'rsaddon' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		$this->add_control(
			'summery_intro_title',
			[
				'label'       => esc_html__( 'Intro Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Summery Intro Title Here..', 'rsaddon' ),
				'default' => 'Summery',
			]
		);
		$this->add_control(
			'out_principal_title',
			[
				'label'       => esc_html__( 'Principal Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Principal Title Here..', 'rsaddon' ),
				'default' => 'Principal',
			]
		);
		$this->add_control(
			'out_interest_title',
			[
				'label'       => esc_html__( 'Interest Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Interest Title Here..', 'rsaddon' ),
				'default' => 'Interest',
			]
		);
		$this->add_control(
			'out_pay_month_title',
			[
				'label'       => esc_html__( 'Monthly Payable Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Monthly Payable Title Here..', 'rsaddon' ),
				'default' => 'Monthly Payable',
			]
		);
		$this->add_control(
			'out_pay_total_title',
			[
				'label'       => esc_html__( 'Total Payable Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Total Payable Title Here..', 'rsaddon' ),
				'default' => 'Total Payable',
			]
		);

		// Chart Area
		$this->add_control(
			'chart_area_title_options',
			[
				'label' => esc_html__( 'Graph Area', 'rsaddon' ),
				'type' => Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);
		$this->add_control(
			'chart_intro_show',
			[
				'label' => esc_html__( 'Chart Intro', 'rsaddon' ),
				'type' => Controls_Manager::SWITCHER,
				'label_on' => esc_html__( 'Show', 'rsaddon' ),
				'label_off' => esc_html__( 'Hide', 'rsaddon' ),
				'return_value' => 'yes',
				'default' => 'yes',
			]
		);
		$this->add_control(
			'chart_intro_title',
			[
				'label'       => esc_html__( 'Intro Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Chart Intro Title Here..', 'rsaddon' ),
				'default' => 'Payment Breakup',
				'condition' => [
					'chart_intro_show' => 'yes'
				]
			]
		);
		$this->add_control(
			'chart_loan_details_title',
			[
				'label'       => esc_html__( 'Loan Details Title', 'rsaddon' ),
				'type'        => Controls_Manager::TEXT,
				'label_block' => true,
				'placeholder' => esc_html__( 'Loan Details Title Here..', 'rsaddon' ),
				'default' => 'Loan Details'
			]
		);
		$this->end_controls_section();
	}
	// Content Dynamic Input End

	// Content Wrapper Style Start
	protected function register_loan_calculator_wrapper_style_controls()
    {
		$this->start_controls_section(
			'_wrapper_style',
		    [
				'label' => esc_html__( 'Wrapper Style', 'rsaddon' ),
		        'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
			$this->add_responsive_control(
				'wrapper_v_align',
				[
					'label' => esc_html__( 'Vertical Align', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'flex-start' => [
							'title' => esc_html__( 'Top', 'rsaddon' ),
							'icon' => 'eicon-align-start-v',
						],
						'center' => [
							'title' => esc_html__( 'Middle', 'rsaddon' ),
							'icon' => 'eicon-align-center-v',
						],
						'flex-end' => [
							'title' => esc_html__( 'Bottom', 'rsaddon' ),
							'icon' => 'eicon-align-end-v',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator' => 'align-items: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'wrapper_column_align',
				[
					'label' => esc_html__( 'Column Direction', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'row' => [
							'title' => esc_html__( 'Row', 'rsaddon' ),
							'icon' => 'eicon-justify-start-h',
						],
						'row-reverse' => [
							'title' => esc_html__( 'Row Reverse', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
						'column' => [
							'title' => esc_html__( 'Column', 'rsaddon' ),
							'icon' => 'eicon-justify-start-v',
						],
						'column-reverse' => [
							'title' => esc_html__( 'Column Reverse', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator' => 'flex-direction: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'wrapper_h_align',
				[
					'label' => esc_html__( 'Horizontal Align', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'flex-start' => [
							'title' => esc_html__( 'Start', 'rsaddon' ),
							'icon' => 'eicon-align-start-h',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'rsaddon' ),
							'icon' => 'eicon-align-center-h',
						],
						'flex-end' => [
							'title' => esc_html__( 'End', 'rsaddon' ),
							'icon' => 'eicon-align-end-h',
						],
						'space-between' => [
							'title' => esc_html__( 'Space Between', 'rsaddon' ),
							'icon' => 'eicon-justify-space-between-h',
						],

					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator' => 'justify-content: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'wrapper_flex_wrap',
				[
					'label' => esc_html__( 'Flex Wrap', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'nowrap' => [
							'title' => esc_html__( 'No Wrap', 'rsaddon' ),
							'icon' => 'eicon-nowrap',
						],
						'wrap' => [
							'title' => esc_html__( 'Wrap', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator' => 'flex-wrap: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'wrapper_flex_gap',
				[
					'label' => esc_html__( 'Gap Between', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator' => 'gap: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'content_side_options',
				[
					'label' => esc_html__( 'Content Side Options', 'rsaddon' ),
					'type' => Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'content_side_width',
				[
					'label' => esc_html__( 'Width', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper' => 'max-width: {{SIZE}}{{UNIT}};',
					],
				]
			);

			$this->add_control(
				'chart_side_options',
				[
					'label' => esc_html__( 'Graph Side Options', 'rsaddon' ),
					'type' => Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_responsive_control(
				'chart_side_width',
				[
					'label' => esc_html__( 'Width', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => '%',
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .chart-wrapper' => 'max-width: {{SIZE}}{{UNIT}};',
					],
				]
			);
		$this->end_controls_section();
	}
	// Content Wrapper Style End

	// Range Slider Style Start
	protected function register_loan_calculator_range_slider_style_controls()
    {
		$this->start_controls_section(
			'_range_slider_style',
		    [
				'label' => esc_html__( 'Range Slider Style', 'rsaddon' ),
		        'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
			$this->add_responsive_control(
				'range_slider_height',
				[
					'label' => esc_html__( 'Height', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'show_label' => true,
					'size_units' => [ '%', 'px' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input' => 'height: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-webkit-slider-runnable-track' => 'height: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-moz-range-track' => 'height: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-webkit-slider-thumb' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',                 
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-moz-range-thumb' => 'width: calc({{SIZE}}{{UNIT}} - 3px); height: calc({{SIZE}}{{UNIT}} - 3px);',                 
					],
				]
			);
			
			$this->add_control(
				'range_slider_bg_color',
				[
					'label' => esc_html__( 'Track Background', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-webkit-slider-runnable-track' => 'background: {{VALUE}}',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-moz-range-track' => 'background: {{VALUE}}',
					],
				]
			);
			$this->add_control(
				'range_slider_bg_color_progress',
				[
					'label' => esc_html__( 'Progress Background', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-webkit-slider-thumb' => 'border: 2px solid {{VALUE}}; box-shadow: -407px 0 0 400px {{VALUE}};',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-moz-range-thumb' => 'border: 2px solid {{VALUE}}; box-shadow: -407px 0 0 400px {{VALUE}};',
					],
				]
			);
			$this->add_control(
				'range_slider_bg_color_progress_dot',
				[
					'label' => esc_html__( 'Progress Dot Background', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-webkit-slider-thumb' => 'background-color: {{VALUE}};',
						'{{WRAPPER}} .rs-addon-loan-calculator .data-slider .data-item input::-moz-range-thumb' => 'background-color: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'range_slider_margin',
				[
					'label' => esc_html__( 'Margin', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'custom' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-slider .data-item input' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'range_slider_border_radius',
				[
					'label' => esc_html__( 'Border Radius', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-slider .data-item input' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			// Slider Label
			$this->add_control(
				'heading_range_slider_label',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Slider Label', 'rsaddon' ),
					'separator' => 'before',
				]
			);
			$this->add_control(
				'range_slider_label_color',
				[
					'label' => esc_html__( 'Text Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-slider .data-item label' => 'color: {{VALUE}}',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'range_slider_label_typography',
					'label' => esc_html__( 'Typography', 'rsaddon' ),
					'selector' => '.rs-addon-loan-calculator .content-wrapper .data-slider .data-item label',
				]
			);
			// Slider Value
			$this->add_control(
				'heading_range_slider_value',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Slider Value', 'rsaddon' ),
					'separator' => 'before',
				]
			);
			$this->add_control(
				'range_slider_value_color',
				[
					'label' => esc_html__( 'Text Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-slider .data-item span' => 'color: {{VALUE}}',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'range_slider_value_typography',
					'label' => esc_html__( 'Typography', 'rsaddon' ),
					'selector' => '.rs-addon-loan-calculator .content-wrapper .data-slider .data-item span',
				]
			);
		$this->end_controls_section();
	}
	// Range Slider Style End

	// Summery Part Style Start
	protected function register_loan_calculator_summery_part_style_controls()
    {
		$this->start_controls_section(
			'_summery_part_style',
		    [
				'label' => esc_html__( 'Summery Part Style', 'rsaddon' ),
		        'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
			// Summery Intro Heading
			$this->add_control(
				'summery_intro_title_heading',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Summery Intro Title', 'rsaddon' ),
				]
			);
			$this->add_control(
				'summery_intro_title_color',
				[
					'label' => esc_html__( 'Text Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .sum_heading' => 'color: {{VALUE}}',
					]
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'summery_intro_title_typography',
					'label' => esc_html__( 'Typography', 'rsaddon' ),
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .sum_heading',
				]
			);
			$this->add_responsive_control(
				'summery_intro_title_margin',
				[
					'label' => esc_html__( 'Margin', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'custom' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .sum_heading' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			// Card Style
			$this->add_control(
				'_summery_card_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Card Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_responsive_control(
				'summery_card_v_align',
				[
					'label' => esc_html__( 'Vertical Align', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'flex-start' => [
							'title' => esc_html__( 'Top', 'rsaddon' ),
							'icon' => 'eicon-align-start-v',
						],
						'center' => [
							'title' => esc_html__( 'Middle', 'rsaddon' ),
							'icon' => 'eicon-align-center-v',
						],
						'flex-end' => [
							'title' => esc_html__( 'Bottom', 'rsaddon' ),
							'icon' => 'eicon-align-end-v',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result' => 'align-items: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_column_align',
				[
					'label' => esc_html__( 'Column Direction', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'row' => [
							'title' => esc_html__( 'Row', 'rsaddon' ),
							'icon' => 'eicon-justify-start-h',
						],
						'row-reverse' => [
							'title' => esc_html__( 'Row Reverse', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
						'column' => [
							'title' => esc_html__( 'Column', 'rsaddon' ),
							'icon' => 'eicon-justify-start-v',
						],
						'column-reverse' => [
							'title' => esc_html__( 'Column Reverse', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result' => 'flex-direction: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_h_align',
				[
					'label' => esc_html__( 'Horizontal Align', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'flex-start' => [
							'title' => esc_html__( 'Start', 'rsaddon' ),
							'icon' => 'eicon-align-start-h',
						],
						'center' => [
							'title' => esc_html__( 'Center', 'rsaddon' ),
							'icon' => 'eicon-align-center-h',
						],
						'flex-end' => [
							'title' => esc_html__( 'End', 'rsaddon' ),
							'icon' => 'eicon-align-end-h',
						],
						'space-between' => [
							'title' => esc_html__( 'Space Between', 'rsaddon' ),
							'icon' => 'eicon-justify-space-between-h',
						],

					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result' => 'justify-content: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_flex_wrap',
				[
					'label' => esc_html__( 'Flex Wrap', 'rsaddon' ),
					'type' => Controls_Manager::CHOOSE,
					'options' => [
						'nowrap' => [
							'title' => esc_html__( 'No Wrap', 'rsaddon' ),
							'icon' => 'eicon-nowrap',
						],
						'wrap' => [
							'title' => esc_html__( 'Wrap', 'rsaddon' ),
							'icon' => 'eicon-wrap',
						],
					],
					'toggle' => true,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result' => 'flex-wrap: {{VALUE}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_flex_gap',
				[
					'label' => esc_html__( 'Gap Between', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result' => 'gap: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_width',
				[
					'label' => esc_html__( 'Card Width', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => '%'
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details' => 'max-width: {{SIZE}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_padding',
				[
					'label' => esc_html__( 'Padding', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'custom' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_responsive_control(
				'summery_card_radius',
				[
					'label' => esc_html__( 'Border Radius', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'custom' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name' => 'summery_card_background',
					'types' => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details',
				]
			);
			$this->add_group_control(
				Group_Control_Border::get_type(),
				[
					'name' => 'summery_card_border',
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details',
				]
			);
			$this->add_group_control(
				Group_Control_Box_Shadow::get_type(),
				[
					'name' => 'summery_card_shadow',
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details',
				]
			);
			$this->add_control(
				'_summery_card_text_label_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Card Label Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_control(
				'summery_card_label_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details label' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'summery_card_label_typography',
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details label',
				]
			);
			$this->add_responsive_control(
				'summery_card_label_margin',
				[
					'label' => esc_html__( 'Margin', 'rsaddon' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', '%', 'custom' ],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details label' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);
			$this->add_control(
				'_summery_card_text_value_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Card Value Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_control(
				'summery_card_value_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details > div' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'summery_card_value_typography',
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details > div',
				]
			);
			
			$this->add_control(
				'_summery_card_principal_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Principal Card Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_responsive_control(
				'principal_order',
				[
					'label' => esc_html__( 'Order', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 1,
							'max' => 8,
						]
					],
					'default' => [
						'size' => 1,
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.principal' => 'order: {{SIZE}};',
					],
				]
			);
			$this->add_control(
				'principal_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.principal' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name' => 'principal_background',
					'types' => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.principal',
				]
			);
			$this->add_control(
				'principal_border_color',
				[
					'label' => esc_html__( 'Border Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.principal' => 'border-color: {{VALUE}}',
					],
				]
			);

			$this->add_control(
				'_summery_card_interest_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Interest Card Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_responsive_control(
				'interest_order',
				[
					'label' => esc_html__( 'Order', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 1,
							'max' => 8,
						]
					],
					'default' => [
						'size' => 2,
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.interest' => 'order: {{SIZE}};',
					],
				]
			);
			$this->add_control(
				'interest_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.interest' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name' => 'interest_background',
					'types' => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.interest',
				]
			);
			$this->add_control(
				'interest_border_color',
				[
					'label' => esc_html__( 'Border Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.interest' => 'border-color: {{VALUE}}',
					],
				]
			);

			$this->add_control(
				'_summery_card_m_payable_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Monthly Payable Card Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_responsive_control(
				'm_payable_order',
				[
					'label' => esc_html__( 'Order', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 1,
							'max' => 8,
						]
					],
					'default' => [
						'size' => 3,
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.m-payable' => 'order: {{SIZE}};',
					],
				]
			);
			$this->add_control(
				'm_payable__color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.m-payable' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name' => 'm_payable_background',
					'types' => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.m-payable',
				]
			);
			$this->add_control(
				'm_payable_border_color',
				[
					'label' => esc_html__( 'Border Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.m-payable' => 'border-color: {{VALUE}}',
					],
				]
			);

			$this->add_control(
				'_summery_card_t_payable_style_h',
				[
					'type' => Controls_Manager::HEADING,
					'label' => esc_html__( 'Total Payable Card Style', 'rsaddon' ),
					'separator' => 'before'
				]
			);
			$this->add_responsive_control(
				't_payable_order',
				[
					'label' => esc_html__( 'Order', 'rsaddon' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px' ],
					'range' => [
						'px' => [
							'min' => 1,
							'max' => 8,
						]
					],
					'default' => [
						'size' => 4,
					],
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.t-payable' => 'order: {{SIZE}};',
					],
				]
			);
			$this->add_control(
				't_payable__color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.t-payable' => 'color: {{VALUE}}',
					],
				]
			);
			$this->add_group_control(
				Group_Control_Background::get_type(),
				[
					'name' => 't_payable_background',
					'types' => [ 'classic', 'gradient' ],
					'selector' => '{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.t-payable',
				]
			);
			$this->add_control(
				't_payable_border_color',
				[
					'label' => esc_html__( 'Border Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .rs-addon-loan-calculator .content-wrapper .data-result .chart-details.t-payable' => 'border-color: {{VALUE}}',
					],
				]
			);
		$this->end_controls_section();
	}
	// Summery Part Style End

	// Chart Part Style Start
	protected function register_loan_calculator_chart_part_style_controls()
    {
		$this->start_controls_section(
			'_chart_part_style',
		    [
				'label' => esc_html__( 'Graph Part Style', 'rsaddon' ),
		        'tab'   => Controls_Manager::TAB_STYLE,
			]
		);
			$this->add_control(
				'principal_graph_options',
				[
					'label' => esc_html__( 'Principal Chart', 'rsaddon' ),
					'type' => Controls_Manager::HEADING
				]
			);
			$this->add_control(
				'principal_graph_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR
				]
			);
			$this->add_control(
				'interest_graph_options',
				[
					'label' => esc_html__( 'Interest Chart', 'rsaddon' ),
					'type' => Controls_Manager::HEADING,
					'separator' => 'before',
				]
			);
			$this->add_control(
				'interest_graph_color',
				[
					'label' => esc_html__( 'Color', 'rsaddon' ),
					'type' => Controls_Manager::COLOR
				]
			);
		$this->end_controls_section();
	}
	// Chart Part Style End

	/**
	 * Render counter widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	/**
	 * Render counter widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$unique = $this->get_id();
		$currency = !empty($settings['currency']) ? $settings['currency'] : '$';
		$chartIntro = !empty($settings['chart_intro_show']) ? 'true' : 'false';
		$primaryColor = !empty($settings['principal_graph_color']) ? $settings['principal_graph_color'] : 'rgb(92, 180, 106)';
		$secondaryColor = !empty($settings['interest_graph_color']) ? $settings['interest_graph_color'] : 'rgb(3, 51, 51)';
		?>

		<div class="rs-addon-loan-calculator">
			<div class="content-wrapper">
				<div class="data-slider">
					<div class="data-item">
						<label><?php echo esc_html( $settings['in_amount_title'] ); ?></label>
						<span id="loan-amt-text-<?php echo esc_attr($unique);?>">0</span>
						<input type="range" id="loan-amount-<?php echo esc_attr($unique);?>" min="1000" max="100000" step="500">
					</div>
					<div class="data-item">
						<label><?php echo esc_html( $settings['in_duration_title'] ); ?></label>
						<span id="loan-period-text-<?php echo esc_attr($unique);?>">0</span>
						<input type="range" id="loan-period-<?php echo esc_attr($unique);?>" min="1" max="60" step="0.5" value="36">
					</div>
					<div class="data-item">
						<label><?php echo esc_html( $settings['in_interest_title'] ); ?></label>
						<span id="interest-rate-text-<?php echo esc_attr($unique);?>">0</span>
						<input type="range" id="interest-rate-<?php echo esc_attr($unique);?>" min="1" max="50" step="0.5" value="4">
					</div>
				</div>
				<?php if (!empty($settings['summery_intro_title'])) {?>
					<h4 class="sum_heading"><?php echo esc_html( $settings['summery_intro_title'] ); ?></h4>
				<?php } ?>
				<div class="data-result">
					<div class='chart-details principal'>
						<label><?php echo esc_html( $settings['out_principal_title'] ); ?></label>
						<div id="cp-<?php echo esc_attr($unique);?>">0</div>
					</div>
					<div class='chart-details interest'>
						<label><?php echo esc_html( $settings['out_interest_title'] ); ?></label>
						<div id="ci-<?php echo esc_attr($unique);?>">0</div>
					</div>
					<div class='chart-details m-payable'>
						<label><?php echo esc_html( $settings['out_pay_month_title'] ); ?></label>
						<div><span id="price-<?php echo esc_attr($unique);?>">0</span>/mo</div>
					</div>
					<div class='chart-details t-payable'>
						<label><?php echo esc_html( $settings['out_pay_total_title'] ); ?></label>
						<div id="ct-<?php echo esc_attr($unique);?>">0</div>
					</div>
				</div>
			</div>
			<div class="chart-wrapper">
				<canvas id="pieChart-<?php echo esc_attr($unique);?>"></canvas>
			</div>
		</div>

		<script>
			jQuery(document).ready(function($) {
				var P, R, N, pie;
				var loan_amt_slider = document.getElementById("loan-amount-<?php echo esc_attr($unique);?>");
				var int_rate_slider = document.getElementById("interest-rate-<?php echo esc_attr($unique);?>");
				var loan_period_slider = document.getElementById("loan-period-<?php echo esc_attr($unique);?>");

				// Update loan amount
				loan_amt_slider.addEventListener("input", (self) => {
					document.querySelector("#loan-amt-text-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseInt(self.target.value).toLocaleString("en-US");
					P = parseFloat(self.target.value);
					displayDetails();
				});

				// Update Rate of Interest
				int_rate_slider.addEventListener("input", (self) => {
					document.querySelector("#interest-rate-text-<?php echo esc_attr($unique);?>").innerText = self.target.value + "%";
					R = parseFloat(self.target.value);
					displayDetails();
				});

				// Update loan period
				loan_period_slider.addEventListener("input", (self) => {
					document.querySelector("#loan-period-text-<?php echo esc_attr($unique);?>").innerText =
						self.target.value + " Months";
					N = parseFloat(self.target.value);
					displayDetails();
				});

				// Calculate total Interest payable
				function calculateLoanDetails(p, r, emi) {
					let totalInterest = 0;
					while (p > 0) {
						let interest = parseFloat(p) * parseFloat(r);
						p = parseFloat(p) - (parseFloat(emi) - interest);
						totalInterest += interest;
					}
					return totalInterest;
				}

				// Calculate and display details
				function displayDetails() {
					let r = parseFloat(R) / 1200;
					let n = parseFloat(N);

					let num = parseFloat(P) * r * Math.pow(1 + r, n);
					let denom = Math.pow(1 + r, n) - 1;
					let emi = parseFloat(num) / parseFloat(denom);
					let payableInterest = calculateLoanDetails(P, r, emi);

					let opts = '{style: "decimal", currency: "US"}';

					// Update principal, interest, and monthly payable values
					document.querySelector("#cp-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseFloat(P).toLocaleString("en-US", opts);

					document.querySelector("#ci-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseFloat(payableInterest).toLocaleString("en-US", opts);

					document.querySelector("#ct-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseFloat(parseFloat(P) + parseFloat(payableInterest)).toLocaleString(
							"en-US",
							opts
						);

					document.querySelector("#price-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseFloat(emi).toLocaleString("en-US", opts);

					// Update pie chart
					pie.data.datasets[0].data[0] = P;
					pie.data.datasets[0].data[1] = payableInterest;
					pie.update();
				}

				// Initialize everything
				function initialize() {
					document.querySelector("#loan-amt-text-<?php echo esc_attr($unique);?>").innerText =
						"<?php echo esc_html($currency); ?>" + ' ' + parseInt(loan_amt_slider.value).toLocaleString("en-US");
					P = parseFloat(document.getElementById("loan-amount-<?php echo esc_attr($unique);?>").value);

					document.querySelector("#interest-rate-text-<?php echo esc_attr($unique);?>").innerText =
						int_rate_slider.value + "%";
					R = parseFloat(document.getElementById("interest-rate-<?php echo esc_attr($unique);?>").value);

					document.querySelector("#loan-period-text-<?php echo esc_attr($unique);?>").innerText =
						loan_period_slider.value + " Months";
					N = parseFloat(document.getElementById("loan-period-<?php echo esc_attr($unique);?>").value);

					pie = new Chart(document.getElementById("pieChart-<?php echo esc_attr($unique);?>"), {
						type: "doughnut",
						data: {
							labels: ["<?php echo esc_html( $settings['out_principal_title'] ); ?>", "<?php echo esc_html( $settings['out_interest_title'] ); ?>"],
							datasets: [
								{
									label: " <?php echo esc_html( $settings['chart_loan_details_title'] ); ?>",
									data: [0, 0],
									backgroundColor: ["<?php echo esc_html($primaryColor); ?>", "<?php echo esc_html($secondaryColor); ?>"],
									hoverOffset: 4
								}
							]
						},
						options: {
							plugins: {
								title: {
									display: <?php echo esc_html( $chartIntro ); ?>,
									text: "<?php echo esc_html( $settings['chart_intro_title'] ); ?>"
								}
							}
						}
					});
					displayDetails();
				}
				
				initialize();
			});
		</script>
	<?php
	}
}